<?php

function import_products_from_csv(PDO $db, $csv_path) {
  if (!file_exists($csv_path)) return [false, "Archivo no encontrado", 0];

  $fh = fopen($csv_path, 'r');
  if (!$fh) return [false, "No se pudo abrir el archivo", 0];

  // Detect delimiter (comma or semicolon)
  $first = fgets($fh);
  if ($first === false) return [false, "CSV vacío", 0];
  $delim = (substr_count($first, ';') > substr_count($first, ',')) ? ';' : ',';
  rewind($fh);

  $headers = fgetcsv($fh, 0, $delim);
  if (!$headers) return [false, "CSV sin encabezados", 0];

  // Normalize headers
  $norm = [];
  foreach ($headers as $i => $h) {
    $h = trim(mb_strtolower($h));
    $h = preg_replace('/\s+/', ' ', $h);
    $norm[$h] = $i;
  }

  // Helper to get column by possible names
  $col = function($names) use ($norm) {
    foreach ($names as $n) {
      $k = trim(mb_strtolower($n));
      if (isset($norm[$k])) return $norm[$k];
    }
    return null;
  };

  $c_sku   = $col(['sku','id','external_id']);
  $c_name  = $col(['name','nombre','productname','title']);
  $c_desc  = $col(['description','descripcion','desc']);
  $c_price = $col(['price','regular price','precio','importe']);
  $c_cat   = $col(['category','categories','categoria','maincatcode']);
  $c_imgs  = $col(['images','image','image_url','imagen','imagenes']);
  $c_color = $col(['color','attribute 1 value(s)']);
  $c_sizes = $col(['sizes','talles','attribute 2 value(s)']);
  $c_stock = $col(['stock','stock quantity','cantidad']);

  if ($c_name === null) return [false, "Falta columna NAME/NOMBRE", 0];

  $count = 0;
  $db->beginTransaction();
  try {
    while (($row = fgetcsv($fh, 0, $delim)) !== false) {
      $name = trim((string)($row[$c_name] ?? ''));
      if ($name === '') continue;

      $external_id = $c_sku !== null ? trim((string)($row[$c_sku] ?? '')) : '';
      $desc = $c_desc !== null ? trim((string)($row[$c_desc] ?? '')) : '';
      $price_raw = $c_price !== null ? trim((string)($row[$c_price] ?? '')) : '';
      // allow "7,99" or "7.99"
      $price_raw = str_replace(['$', 'ARS', ' '], '', $price_raw);
      $price_raw = str_replace(',', '.', $price_raw);
      $price = is_numeric($price_raw) ? (float)$price_raw : 0;

      $cat_name = $c_cat !== null ? trim((string)($row[$c_cat] ?? '')) : '';
      $cat_id = null;
      if ($cat_name !== '') {
        $slug = slugify($cat_name);
        $st = $db->prepare("SELECT id FROM categories WHERE slug=?");
        $st->execute([$slug]);
        $cat_id = $st->fetchColumn();
        if (!$cat_id) {
          $ins = $db->prepare("INSERT INTO categories(name, slug, sort_order) VALUES(?,?,?)");
          $ins->execute([$cat_name, $slug, 50]);
          $cat_id = (int)$db->lastInsertId();
        }
      }

      $slug_base = slugify($name);
      $slug = $slug_base;
      $i = 2;
      while (true) {
        $st = $db->prepare("SELECT id FROM products WHERE slug=?");
        $st->execute([$slug]);
        $existing_id = $st->fetchColumn();
        if (!$existing_id) break;

        if ($external_id) {
          $st2 = $db->prepare("SELECT id FROM products WHERE external_id=?");
          $st2->execute([$external_id]);
          $eid = $st2->fetchColumn();
          if ($eid && (int)$eid === (int)$existing_id) break;
        }
        $slug = $slug_base . '-' . $i; $i++;
      }

      $pid = null;
      if ($external_id) {
        $st = $db->prepare("SELECT id FROM products WHERE external_id=?");
        $st->execute([$external_id]);
        $pid = $st->fetchColumn();
      }
      if (!$pid) {
        $st = $db->prepare("SELECT id FROM products WHERE slug=?");
        $st->execute([$slug]);
        $pid = $st->fetchColumn();
      }

      $now = gmdate('c');
      if ($pid) {
        $up = $db->prepare("UPDATE products SET name=?, slug=?, description=?, price=?, category_id=?, is_active=1, updated_at=? WHERE id=?");
        $up->execute([$name, $slug, $desc, $price, $cat_id, $now, $pid]);
      } else {
        $ins = $db->prepare("INSERT INTO products(external_id, name, slug, description, price, category_id, is_active, created_at, updated_at)
          VALUES(?,?,?,?,?,?,?,?,?)");
        $ins->execute([$external_id ?: null, $name, $slug, $desc, $price, $cat_id, 1, $now, $now]);
        $pid = (int)$db->lastInsertId();
      }

      // Images: accept pipe/comma separated
      $db->prepare("DELETE FROM product_images WHERE product_id=?")->execute([$pid]);
      $img_raw = $c_imgs !== null ? trim((string)($row[$c_imgs] ?? '')) : '';
      $img_list = [];
      if ($img_raw !== '') {
        $img_list = preg_split('/\s*\|\s*|\s*,\s*/', $img_raw);
      }
      $insimg = $db->prepare("INSERT INTO product_images(product_id, url, sort_order) VALUES(?,?,?)");
      $order = 0;
      foreach ($img_list as $img) {
        $img = trim((string)$img);
        if ($img === '') continue;
        $insimg->execute([$pid, $img, $order++]);
        if ($order >= 8) break;
      }

      // Variants: optional (basic)
      $db->prepare("DELETE FROM variants WHERE product_id=?")->execute([$pid]);
      $color = $c_color !== null ? trim((string)($row[$c_color] ?? '')) : '';
      $sizes = $c_sizes !== null ? trim((string)($row[$c_sizes] ?? '')) : '';
      $stock = $c_stock !== null ? (int)preg_replace('/[^0-9\-]/', '', (string)($row[$c_stock] ?? '0')) : 0;

      if ($sizes !== '') {
        $sz_list = preg_split('/\s*,\s*/', $sizes);
        $insv = $db->prepare("INSERT INTO variants(product_id, sku, color, size, price, stock) VALUES(?,?,?,?,?,?)");
        foreach ($sz_list as $sz) {
          $sz = trim($sz);
          if ($sz === '') continue;
          $insv->execute([$pid, $external_id, $color, $sz, null, max(0,$stock)]);
        }
      } elseif ($stock > 0 || $color !== '') {
        $insv = $db->prepare("INSERT INTO variants(product_id, sku, color, size, price, stock) VALUES(?,?,?,?,?,?)");
        $insv->execute([$pid, $external_id, $color, '', null, max(0,$stock)]);
      }

      $count++;
    }
    $db->commit();
  } catch (Exception $e) {
    $db->rollBack();
    fclose($fh);
    return [false, $e->getMessage(), 0];
  }

  fclose($fh);
  return [true, "OK", $count];
}
