<?php

function http_get_json($url, $headers=[]) {
  $ch = curl_init($url);
  curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
  curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
  curl_setopt($ch, CURLOPT_TIMEOUT, 20);
  if ($headers && is_array($headers)) {
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
  }
  $resp = curl_exec($ch);
  $err = curl_error($ch);
  $code = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
  curl_close($ch);

  if ($resp === false) return [false, "cURL error: $err", null];
  if ($code < 200 || $code >= 300) return [false, "HTTP $code", null];

  $data = json_decode($resp, true);
  if ($data === null) return [false, "JSON inválido", null];
  return [true, "OK", $data];
}

// Flexible mapper for common API shapes
function normalize_api_items($data) {
  if (is_array($data)) {
    // array of items or object
    $is_assoc = array_keys($data) !== range(0, count($data)-1);
    if (!$is_assoc) return $data;
    if (isset($data['items']) && is_array($data['items'])) return $data['items'];
    if (isset($data['data']) && is_array($data['data'])) return $data['data'];
    if (isset($data['results']) && is_array($data['results'])) return $data['results'];
  }
  return [];
}

function pick($arr, $keys, $default=null) {
  foreach ($keys as $k) {
    if (isset($arr[$k]) && $arr[$k] !== '') return $arr[$k];
  }
  return $default;
}

function import_products_from_api(PDO $db, $config, $api_url) {
  if (!$api_url) return [false, "Falta URL de API", 0];

  $headers = $config['products_api_headers'] ?? [];
  [$ok, $msg, $data] = http_get_json($api_url, $headers);
  if (!$ok) return [false, $msg, 0];

  $items = normalize_api_items($data);
  if (!$items) return [false, "La API no devolvió items", 0];

  $count = 0;

  $db->beginTransaction();
  try {
    foreach ($items as $it) {
      if (!is_array($it)) continue;

      $external_id = (string)pick($it, ['id','external_id','product_id','sku_id'], null);
      $name = (string)pick($it, ['name','title','productName'], '');
      if ($name === '') continue;

      $desc = (string)pick($it, ['description','desc','details'], '');
      $price = (float)pick($it, ['price','amount','sale_price','current_price'], 0);

      $cat_name = (string)pick($it, ['category','category_name','collection'], '');
      $cat_id = null;
      if ($cat_name !== '') {
        $slug = slugify($cat_name);
        $st = $db->prepare("SELECT id FROM categories WHERE slug=?");
        $st->execute([$slug]);
        $cat_id = $st->fetchColumn();
        if (!$cat_id) {
          $ins = $db->prepare("INSERT INTO categories(name, slug, sort_order) VALUES(?,?,?)");
          $ins->execute([$cat_name, $slug, 50]);
          $cat_id = (int)$db->lastInsertId();
        }
      }

      $slug = slugify($name);
      // ensure unique slug
      $try = $slug; $i = 2;
      while (true) {
        $st = $db->prepare("SELECT id FROM products WHERE slug=?");
        $st->execute([$try]);
        $existing_id = $st->fetchColumn();
        if (!$existing_id) break;
        // if same external id, allow overwrite
        if ($external_id) {
          $st2 = $db->prepare("SELECT id FROM products WHERE external_id=?");
          $st2->execute([$external_id]);
          $eid = $st2->fetchColumn();
          if ($eid && (int)$eid === (int)$existing_id) break;
        }
        $try = $slug . '-' . $i; $i++;
      }
      $slug = $try;

      // Upsert product by external_id if present, else by slug
      $pid = null;
      if ($external_id) {
        $st = $db->prepare("SELECT id FROM products WHERE external_id=?");
        $st->execute([$external_id]);
        $pid = $st->fetchColumn();
      }
      if (!$pid) {
        $st = $db->prepare("SELECT id FROM products WHERE slug=?");
        $st->execute([$slug]);
        $pid = $st->fetchColumn();
      }

      $now = gmdate('c');
      if ($pid) {
        $up = $db->prepare("UPDATE products SET name=?, slug=?, description=?, price=?, category_id=?, is_active=1, updated_at=? WHERE id=?");
        $up->execute([$name, $slug, $desc, $price, $cat_id, $now, $pid]);
      } else {
        $ins = $db->prepare("INSERT INTO products(external_id, name, slug, description, price, category_id, is_active, created_at, updated_at)
          VALUES(?,?,?,?,?,?,?,?,?)");
        $ins->execute([$external_id, $name, $slug, $desc, $price, $cat_id, 1, $now, $now]);
        $pid = (int)$db->lastInsertId();
      }

      // Images
      $db->prepare("DELETE FROM product_images WHERE product_id=?")->execute([$pid]);
      $images = pick($it, ['images','image_urls','photos'], []);
      $single = pick($it, ['image','image_url','thumbnail'], null);
      $img_list = [];
      if (is_array($images)) $img_list = $images;
      if ($single && !in_array($single, $img_list)) array_unshift($img_list, $single);

      // Some APIs use objects: [{url:"..."}]
      $insimg = $db->prepare("INSERT INTO product_images(product_id, url, sort_order) VALUES(?,?,?)");
      $order = 0;
      foreach ($img_list as $img) {
        if (is_array($img)) $img = $img['url'] ?? ($img['src'] ?? '');
        $img = trim((string)$img);
        if ($img === '') continue;
        $insimg->execute([$pid, $img, $order++]);
        if ($order >= 8) break;
      }

      // Variants
      $db->prepare("DELETE FROM variants WHERE product_id=?")->execute([$pid]);
      $variants = pick($it, ['variants','options'], []);
      if (is_array($variants) && count($variants) > 0) {
        $insv = $db->prepare("INSERT INTO variants(product_id, sku, color, size, price, stock) VALUES(?,?,?,?,?,?)");
        foreach ($variants as $v) {
          if (!is_array($v)) continue;
          $sku = (string)pick($v, ['sku','id','variant_id'], '');
          $color = (string)pick($v, ['color','colour'], '');
          $size = (string)pick($v, ['size','talle'], '');
          $vprice = pick($v, ['price','amount'], null);
          $vprice = ($vprice === null || $vprice === '') ? null : (float)$vprice;
          $stock = (int)pick($v, ['stock','qty','quantity'], 0);
          $insv->execute([$pid, $sku, $color, $size, $vprice, $stock]);
        }
      } else {
        // Optional basic stock from API
        $stock = (int)pick($it, ['stock','qty','quantity'], 0);
        if ($stock > 0) {
          $insv = $db->prepare("INSERT INTO variants(product_id, sku, color, size, price, stock) VALUES(?,?,?,?,?,?)");
          $insv->execute([$pid, (string)$external_id, '', '', null, $stock]);
        }
      }

      $count++;
    }

    $db->commit();
  } catch (Exception $e) {
    $db->rollBack();
    return [false, $e->getMessage(), 0];
  }

  return [true, "OK", $count];
}
